unit StProgressIndicator;

interface

uses
  SysUtils, Classes, Graphics, Controls, Forms, ComCtrls, ExtCtrls, Gauges;

type
  TTimeStatusBar = class(TStatusBar)
  private
    FElapsedText: String;
    FEstimatedText: String;
    FText1: String;
    FText2: String;
    FCleared: Boolean;
    FTextAlign: TAlignment;
    procedure SetElapsedText(const Value: String);
    procedure SetEstimatedText(const Value: String);
    procedure SetText1(const Value: String);
    procedure SetText2(const Value: String);
    procedure SetCleared(const Value: Boolean);
    procedure SetTextAlign(const Value: TAlignment);
    { Private declarations }
  protected
    { Protected declarations }
    procedure SetTitles;
    procedure Resize; override;
    procedure Click; override;
    procedure DblClick; override;
  public
    { Public declarations }
     constructor Create(AOwner: TComponent); override;
  published
    { Published declarations }
   
    property ElapsedText: String read FElapsedText write SetElapsedText;
    property EstimatedText: String read FEstimatedText write SetEstimatedText;
    property Text1: String read FText1 write SetText1;
    property Text2: String read FText2 write SetText2;
    property Cleared: Boolean read FCleared write SetCleared;
    property TextAlign: TAlignment read FTextAlign write SetTextAlign default taLeftJustify;
  end;

type TTextPosition = (tpTop, tpBottom, tpLeft, tpRight);

type
  TMyProgressBar = Class(TGauge)
  protected
    procedure Click; override;
    procedure DblClick; override;
  end;

type
  TStProgressIndicator = class(TCustomPanel)
  private
    { Private declarations }
    FStatus: TTimeStatusBar;
    FTimer: TTimer;
    FProgressBar: TMyProgressBar;
    FTextPosition: TTextPosition;
    FStopedVisible: Boolean;
    FOnStop, FOnStart: TNotifyEvent;
    FStartValue, FStoptValue, FRange, FCurrentValue: Int64;
    FStartTime: TDateTime;
    FAutostop: Boolean;
    function GetTextAlign: TAlignment;
    procedure SetTextAlign(const Value: TAlignment);
    procedure SetTextPosition(const Value: TTextPosition);
    function GetElapsedText: String;
    procedure SetElapsedText(const Value: String);
    function GetEstimatedText: String;
    procedure SetEstimatedText(const Value: String);
    function GetActive: Boolean;
    procedure SetStopedVisible(const Value: Boolean);
    procedure SetCurrentValue(const Value: Int64);
    function GetForeColor: TColor;
    procedure SetForeColor(const Value: TColor);
    function GetBackColor: TColor;
    procedure SetBackColor(const Value: TColor);
    function GetShowPercent: Boolean;
    procedure SetShowPercent(const Value: Boolean);
    function GetBgColor: TColor;
    procedure SetBgColor(const Value: TColor);

  protected
    { Protected declarations }
    procedure Resize; override;
    procedure FTimerTimer(Sender: TObject); dynamic;
  public
    { Public declarations }
    constructor Create(AOwner: TComponent); override;
    property CurrentValue: Int64 read FCurrentValue write SetCurrentValue;
    property Active: Boolean read GetActive;
    procedure Stop;
    procedure Start(const StartValue, EndValue: Int64);
  published
    { Published declarations }
    property TextAlign: TAlignment read GetTextAlign write SetTextAlign;
    property TextPosition: TTextPosition read FTextPosition
      write SetTextPosition default tpBottom;
    property ElapsedText: String read GetElapsedText write SetElapsedText;
    property EstimatedText: String read GetEstimatedText
      write SetEstimatedText;
    property StopedVisible: Boolean read FStopedVisible
      write SetStopedVisible default false;

    property ForeColor: TColor read GetForeColor write SetForeColor
      default clBlue;
    property BackColor: TColor read GetBackColor write SetBackColor
      default clWhite;
    property Autostop: Boolean read FAutostop write FAutostop default true;
    property ShowPercent: Boolean read GetShowPercent write SetShowPercent
      default true;

    property BgColor: TColor read GetBgColor write SetBgColor default
      clBtnFace;
    property OnStop: TNotifyEvent read FOnStop write FOnStop;
    property OnStart: TNotifyEvent read FOnStart write FOnStart;
    property OnClick;
    property OnDblClick;

    property Width default 250;
    property Height default 40;

    property Align;
    property Alignment;
    property Anchors;
    property AutoSize;
    property BevelInner;
    property BevelOuter;
    property BevelWidth;
    property BorderWidth;
    property BorderStyle;
    property Constraints;
    property Ctl3D;
    property Enabled;
    property ParentCtl3D;
    property ParentShowHint;
    property PopupMenu;
    property ShowHint;
    property TabOrder;
    property TabStop;
    property Visible;
    property OnCanResize;
    property OnConstrainedResize;
    property OnContextPopup;
    property OnGetSiteInfo;
    property OnResize;

  end;

procedure Register;

implementation
uses dialogs;  //dialogs  

const
   minTextHeight = 15;
   minTextWidth = 70;

{ TTimeStatusBar }

procedure TTimeStatusBar.Click;
var
  ce: TNotifyEvent;
begin
  inherited;
  ce := TStProgressIndicator(OWner).OnClick;
  if Assigned(ce) then ce(Owner);
end;

constructor TTimeStatusBar.Create(AOwner: TComponent);
begin
  inherited;
  SizeGrip:=False;
  Panels.Add;
  Panels.Add;

  FElapsedText := 'Elapsed time';
  FEstimatedText := 'Estimated';

  Constraints.MinHeight := minTextHeight;
  Constraints.MinWidth := minTextWidth;
end;


procedure TTimeStatusBar.DblClick;
var
  ce: TNotifyEvent;
begin
  inherited;
  ce := TStProgressIndicator(OWner).OnDblClick;
  if Assigned(ce) then ce(Owner);
end;

procedure TTimeStatusBar.Resize;
begin
  inherited;
  Panels[0].Width := Width div 2;
  Panels[1].Width := Width div 2;
  SetTitles;
end;

procedure TTimeStatusBar.SetCleared(const Value: Boolean);
begin
  FCleared := Value;
  SetTitles;
end;

procedure TTimeStatusBar.SetElapsedText(const Value: String);
begin
  FElapsedText := Value;

  if FElapsedText<>'' then
    begin
    While Pos('  ', FElapsedText)<>0 do
       FElapsedText := StringReplace(FElapsedText, '  ',
        ' ', [rfReplaceAll]);

    if FElapsedText[Length(FElapsedText)] = ' ' then
      Delete(FElapsedText, Length(FElapsedText), 1);
    end;
    
  SetTitles;
end;

procedure TTimeStatusBar.SetEstimatedText(const Value: String);
begin
  FEstimatedText := Value;

  if FEstimatedText <> '' then
    begin
    While Pos('  ', FEstimatedText)<>0 do
       FEstimatedText := StringReplace(FEstimatedText, '  ', ' ', [rfReplaceAll]);

    if FEstimatedText[Length(FEstimatedText)] = ' ' then
      Delete(FEstimatedText, Length(FEstimatedText), 1);
    end;

  SetTitles;
end;

procedure TTimeStatusBar.SetText1(const Value: String);
begin
  FText1 := Value;
  SetTitles;
end;

procedure TTimeStatusBar.SetText2(const Value: String);
begin
  FText2 := Value;
  SetTitles;
end;

procedure TTimeStatusBar.SetTextAlign(const Value: TAlignment);
begin
  FTextAlign := Value;
  Panels[0].Alignment := Value;
  Panels[1].Alignment := Value;
end;

procedure TTimeStatusBar.SetTitles;
var
  s1, s2: String;
  w1, w2, w: Integer;
begin
if not FCleared then
  begin
  w := Width div 2 - 5;

  s1 := FElapsedText+' '+FText1;
  s2 := FEstimatedText+' '+FText2;

  w1 := Canvas.TextWidth(s1);
  w2 := Canvas.TextWidth(s2);

  if w1>w then s1:=FText1;
  if w2>w then s2:=FText2;

  Panels[0].Text := s1;
  Panels[1].Text := s2;
  end
else
  begin
  Panels[0].Text := '';
  Panels[1].Text := '';
  end;
end;

{end of TimeStatusBar}
procedure Register;
begin
  RegisterComponents('mine', [TStProgressIndicator]);
end;

{ TStProgressIndicator }

constructor TStProgressIndicator.Create(AOwner: TComponent);
begin
  inherited;
  Width := 250;
  Height := 40;
  FAutostop := True;

  FTextPosition := tpBottom;
  FStatus := TTimeStatusBar.Create(Self);
  FStatus.Parent := Self;
  FStatus.Align:= alBottom;

  FTimer := TTimer.Create(Self);
  FTimer.Interval := 1000;
  FTimer.Enabled := False;
  FTimer.OnTimer := FTimerTimer;

  FProgressBar := TMyProgressBar.Create(Self);
  FProgressBar.Parent := Self;
  FProgressBar.Align := alClient;
  FProgressBar.ForeColor := clBlue;
  FProgressBar.BackColor := clWhite;
  FProgressBar.MinValue := 0;
  FProgressBar.MaxValue := 100;
  FProgressBar.ShowText := True;

  FStopedVisible := False;
  Visible := FStopedVisible;
end;


procedure TStProgressIndicator.FTimerTimer(Sender: TObject);
var
  x: extended;
  done, rest, Now: TDateTime;
  percent: Integer;
begin
  if CurrentValue = FStoptValue then
    if FAutostop then
      begin
      Stop;
      Exit;
      end;

  Now := Date+Time;
  done := Now - FStartTime;
  FStatus.Text1:=TimeToStr(done);
  x := (FCurrentValue-FStartValue)/FRange;

  if x = 0 then
    begin
    FProgressBar.Progress := 0;
    FStatus.Text2 := '';
    end
  else
    begin
    rest := (1-x)/x*done;
    FStatus.Text2 := TimeToStr(rest);
    percent := Round(x*100);
    if FProgressBar.Progress <> percent then
      FProgressBar.Progress := percent;
    end;

  Application.ProcessMessages;
end;

function TStProgressIndicator.GetActive: Boolean;
begin
  Result := FTimer.Enabled;
end;

function TStProgressIndicator.GetBackColor: TColor;
begin
  Result := FProgressBar.BackColor;
end;

function TStProgressIndicator.GetBgColor: TColor;
begin
  Result := FStatus.Color;
end;

function TStProgressIndicator.GetElapsedText: String;
begin
  Result := FStatus.ElapsedText
end;

function TStProgressIndicator.GetEstimatedText: String;
begin
  Result := FStatus.ElapsedText;
end;


function TStProgressIndicator.GetForeColor: TColor;
begin
  Result := FProgressBar.ForeColor;
end;

function TStProgressIndicator.GetShowPercent: Boolean;
begin
  Result := FProgressBar.ShowText;
end;

function TStProgressIndicator.GetTextAlign: TAlignment;
begin
  Result := FStatus.TextAlign
end;

procedure TStProgressIndicator.Resize;
begin
  inherited;
  TextPosition := TextPosition;
end;


procedure TStProgressIndicator.SetBackColor(const Value: TColor);
begin
  FProgressBar.BackColor := Value;
end;

procedure TStProgressIndicator.SetBgColor(const Value: TColor);
begin
  FStatus.Color := Value;
end;

procedure TStProgressIndicator.SetCurrentValue(const Value: Int64);
begin
  FCurrentValue := Value;
  Application.ProcessMessages;
end;

procedure TStProgressIndicator.SetElapsedText(const Value: String);
begin
  If FStatus.ElapsedText <> Value then
    FStatus.ElapsedText := Value;
end;

procedure TStProgressIndicator.SetEstimatedText(const Value: String);
begin
  If FStatus.ElapsedText <> Value then
    FStatus.ElapsedText := Value;
end;


procedure TStProgressIndicator.SetForeColor(const Value: TColor);
begin
  FProgressBar.ForeColor := Value;
end;


procedure TStProgressIndicator.SetShowPercent(const Value: Boolean);
begin
  FProgressBar.ShowText := Value;
end;

procedure TStProgressIndicator.SetStopedVisible(const Value: Boolean);
begin
  FStopedVisible := Value;
  if not Active then Visible := Value;
end;

procedure TStProgressIndicator.SetTextAlign(const Value: TAlignment);
begin
   FStatus.TextAlign := Value
end;



procedure TStProgressIndicator.SetTextPosition(const Value: TTextPosition);
begin
  FTextPosition := Value;
  case Value of
    tpLeft: FStatus.Align := alLeft;
    tpRight: FStatus.Align := alRight;
    tpTop: FStatus.Align := alTop;
    tpBottom: FStatus.Align := alBottom;
  end;

  if FTextPosition in [tpTop, tpBottom] then
     begin
     FStatus.Height := ClientHeight div 2;
     FStatus.Width := ClientWidth;
     Constraints.MinHeight := 2*minTextHeight;
     Constraints.MinWidth := minTextWidth;
     end
  else
     begin
     FStatus.Height := ClientHeight;
     FStatus.Width := Width div 2;
     Constraints.MinHeight := minTextHeight;
     Constraints.MinWidth := 2*minTextWidth;
     end;
end;

procedure TStProgressIndicator.Start(const StartValue, EndValue: Int64);
begin
  if StartValue <= EndValue then
    begin
    If Assigned(FOnStart) then FOnStart(Self);
    Visible := True;
    FStartTime := Date + Time;
    FTimer.Enabled := True;
    FStartValue := StartValue;
    FStoptValue := EndValue;
    FRange := EndValue - StartValue;
    if FRange = 0 then FRange := 1;
    end;
end;

procedure TStProgressIndicator.Stop;
begin
  FTimer.Enabled := false;
  FProgressBar.Progress := 0;
  FStatus.Cleared := True;
  Visible := FStopedVisible;
  if Assigned(FOnStop) then FOnStop(Self);
end;

{ TMyProgressBar }

procedure TMyProgressBar.Click;
var
  ce: TNotifyEvent;
begin
  inherited;
  ce := TStProgressIndicator(OWner).OnClick;
  if Assigned(ce) then ce(Owner);
end;


procedure TMyProgressBar.DblClick;
var
  ce: TNotifyEvent;
begin
  inherited;
  ce := TStProgressIndicator(OWner).OnDblClick;
  if Assigned(ce) then ce(Owner);
end;

end.
