{
 *******************************************************************************
 *                                                                             *
 *  Project: System           /  RRR             Copyright (c) 1998            *
 *  Module:  CFlEdit         /  R    R                                         *
 *  Version: 1.0            /  R     R           Horst Reichert                *
 *  Date:    28.10.2002    /  R R R R    t       Mendelssohnstr. 32            *
 *                        /  R   R     ttttt     D-65817 Eppstein              *
 *                       /  R     R     t        Fax: 06198 500478/501537      *
 *                      /  R       R    t        horst.reichert@rt-science.de  *
 *                                                                             *
 * ____________________________________________________________________________*
 *                                                                             *
 *  Component for selection of colors from a palette                           *
 *  The TColorPickCombo component is distributed as freeware also for          *
 *  commercial use. The source is included with the zipped download file.      *
 *  Please give a proper credit in the about box of your programs or note in   *
 *  your manuals.                                                              *
 *******************************************************************************
}

unit RtColorPicker;

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms,
  Dialogs, Buttons, ExtCtrls, StdCtrls;

const
   NormalColorsCnt=56;
   SystemColorsCnt=27;

type
  TColBtn = class(TSpeedButton)
  private
    FSqareColor:TColor;
  procedure SetSqareColor(Value:TColor);
  protected
  public
    property Canvas;
    constructor Create(AOwner: TComponent); override;
  published
    property SqareColor:TColor read FSqareColor write SetSqareColor;
  end;

  TColorPickCombo=class(TCustomComboBox)
  private
    FActiveColor,FDropDnColor:TColor;
    FOnChanged:TNotifyEvent;
    FOtherBtnCaption:TCaption;
    FItems:TStringlist;
    procedure SetActiveColor(Value:TColor);
    procedure MyDropDown(Sender:TObject);
  protected
    procedure DrawItem(Index:Integer;Rect:TRect;State:TOwnerDrawState); override;
    procedure CreateWnd; override;
    procedure SetBounds(ALeft, ATop, AWidth, AHeight: Integer); override;
    procedure AdjustSize (var W: Integer; var H: Integer);
  public
    constructor Create(AOwner: TComponent); override;
  published
    property ActiveColor:TColor read FActiveColor write SetActiveColor;
    property DropDnColor:TColor read FDropDnColor write FDropDnColor;
    property OtherBtnCaption:TCaption read FOtherBtnCaption write FOtherBtnCaption;
    property OnChanged:TNotifyEvent read FOnChanged write FOnChanged;
    property Enabled;
    property Hint;
    property ShowHint;
    property Visible;
    property Constraints;
    property Anchors;
    property ParentShowHint;
    property PopupMenu;
    property BevelInner;
    property BevelOuter;
    property BevelWidth;
    property BorderWidth;
    property Color;
  end;

  TColorPickerForm = class(TForm)
    Shape1: TShape;
    ColorDialog: TColorDialog;
    procedure FormCreate(Sender: TObject);
    procedure FormDestroy(Sender: TObject);
    procedure FormDeactivate(Sender: TObject);
    procedure FormKeyPress(Sender: TObject; var Key: Char);
  private
    { Private declarations }
    OtherColBtn:TColBtn;
    ColBtns: array[1..NormalColorsCnt+SystemColorsCnt] of TColBtn;
    OtherBtn:TSpeedButton;
    FPickColor:TColor;
    FDoingDialog:Boolean;
    AssignedColorPickCombo:TColorPickCombo;
    procedure OtherBtnClick(Sender:TObject);
    procedure BtnClick(Sender:TObject);
    procedure SetPickColor(Value:TColor);
  public
    { Public declarations }
    property PickColor:TColor read FPickColor write SetPickColor;
  end;

procedure Register;

var
  ColorPickerForm: TColorPickerForm;

implementation

uses Registry;

{$R *.dfm}
const
   BtnDim=20;
   BtnPicDim=14;
   {$IFDEF VER130}
    clGradientActiveCaption = TColor(27 or $80000000);
    clGradientInactiveCaption = TColor(28 or $80000000);
   {$ENDIF}

   BtnColors:array[1..NormalColorsCnt+SystemColorsCnt] of TColor
     =($000000,$808080,$000040,$004040,$004000,$404000,$400000,$400040,
       $202020,$909090,$000080,$008080,$008000,$808000,$800000,$800080,
       $303030,$A0A0A0,$0000C0,$00C0C0,$00C000,$C0C000,$C00000,$C000C0,
       $404040,$B0B0B0,$0000FF,$00FFFF,$00FF00,$FFFF00,$FF0000,$FF00FF,
       $505050,$C0C0C0,$4040FF,$40FFFF,$40FF40,$FFFF40,$FF4040,$FF40FF,
       $606060,$D0D0D0,$8080FF,$80FFFF,$80FF80,$FFFF80,$FF8080,$FF80FF,
       $707070,$FFFFFF,$C0C0FF,$C0FFFF,$C0FFC0,$FFFFC0,$FFC0C0,$FFC0FF,
       clActiveBorder,clActiveCaption,clAppWorkSpace,
       clBackground,clBtnFace,clBtnHighlight,clBtnShadow,clBtnText,clCaptionText,
       clGradientActiveCaption,clGradientInactiveCaption,clGrayText,
       clHighlight,clHighlightText,clInactiveBorder,clInactiveCaption,
       clInactiveCaptionText,clInfoBk,clInfoText,clMenu,clMenuText,
       clScrollBar,cl3DDkShadow,cl3DLight,clWindow,clWindowFrame,clWindowText);

   SysColorsNames:array[1..SystemColorsCnt] of string
     =('Active Border','Active Caption','Application Workspace',
       'Background','Button Face','Button Highlight','Button Shadow',
       'ButtonText','Caption Text','Gradient active Caption',
       'Gradient inactive Caption','Gray Text','Highlight','Highlight Text',
       'Inactive Border','Inactive Caption','Inactive Caption Text','Info Back',
       'Info Text','Menu','Menu Text','Scroll Bar','3D Shadow','3D Light',
       'Window','Window Frame','Window Text');

procedure Register;
begin
  RegisterComponents('Rt-Components', [TColorPickCombo]);
end;

constructor TColBtn.Create(AOwner: TComponent);
begin
  inherited;
  Glyph:=TBitmap.Create;
  Glyph.Width:=BtnPicDim;
  Glyph.Height:=BtnPicDim;
  Glyph.TransparentMode:=tmFixed;
  Glyph.TransparentColor:=Color;
  NumGlyphs:=1;
  Margin:=-1{(BtnDim-BtnPicDim) div 2};
end;

procedure TColBtn.SetSqareColor(Value:TColor);
begin
  FSqareColor:=Value;
  if FSqareColor=Color then Glyph.TransparentColor:=not Color;
  with Glyph.Canvas do
  begin
    Pen.color:=clGray;
    Brush.Color:=FSqareColor;
    Brush.Style:=bsSolid;
    FillRect(Rect(0,0,BtnPicDim,BtnPicDim));
    Brush.Color:=clGray;
    FrameRect(Rect(0,0,BtnPicDim,BtnPicDim));
  end;
end;

procedure TColorPickerForm.FormCreate(Sender: TObject);
var
  i:integer;
  Btn:TColBtn;
  ABtn:TSpeedButton;
  X,Y:Integer;
  dIni:TRegistryIniFile;
begin
  try
    dIni:=TRegistryIniFile.Create('RtColorPicker');
    with dIni do
      for i:=0 to 15 do
        ColorDialog.CustomColors.Add('Color'+Char(65+i)+'='+
                             ReadString('CustomColors','Color'+Char(65+i),''));
  finally
    dIni.Free;
  end;

  for i:=1 to NormalColorsCnt do
  begin
    Btn:=TColBtn.Create(Self);
    Btn.Parent:=Self;
    Btn.Flat:=true;
    Btn.SqareColor:=BtnColors[I];
    Btn.Anchors:=[akBottom];
    Btn.GroupIndex:=1;
    Btn.OnClick:=BtnClick;
    X:=5+((i-1) mod 8)*BtnDim;
    Y:=5+BtnDim*((i-1) div 8);
    Btn.SetBounds(X,Y,BtnDim,BtnDim);
    ColBtns[i]:=Btn;
  end;
  for i:=1 to SystemColorsCnt do
  begin
    Btn:=TColBtn.Create(Self);
    Btn.Parent:=Self;
    Btn.Flat:=true;
    Btn.SqareColor:=BtnColors[NormalColorsCnt+I];
    Btn.Hint:=SysColorsNames[i];
    Btn.ShowHint:=True;
    Btn.Anchors:=[akBottom];
    Btn.GroupIndex:=1;
    Btn.OnClick:=BtnClick;
    X:=170+((i-1) mod 4)*BtnDim;
    Y:=5+BtnDim*((i-1) div 4);
    Btn.SetBounds(X,Y,BtnDim,BtnDim);
    ColBtns[NormalColorsCnt+i]:=Btn;
  end;

  Btn:=TColBtn.Create(Self);
  Btn.Parent:=Self;
  Btn.Flat:=true;
  Btn.SqareColor:=ColorDialog.Color;
  Btn.SetBounds(5,BtnDim*7+10,BtnDim,BtnDim);
  Btn.GroupIndex:=1;
  Btn.Anchors:=[akBottom];
  Btn.OnClick:=BtnClick;
  Btn.Hint:='Custom Color, from standard Windows Color Dialogue';
  Btn.ShowHint:=True;
  OtherColBtn:=Btn;
  ABtn:=TSpeedButton.Create(Self);
  ABtn.Parent:=Self;
  ABtn.Flat:=true;
  ABtn.SetBounds(8+BtnDim,BtnDim*7+10,Width-13-BtnDim,BtnDim);
  ABtn.Anchors:=[akBottom];
  ABtn.Hint:='Open standard Windows Color Dialogue';
  ABtn.ShowHint:=True;
  OtherBtn:=ABtn;
  OtherBtn.OnClick:=OtherBtnClick;
end;

procedure TColorPickerForm.FormDestroy(Sender: TObject);
var dIni:TRegistryIniFile;
    i:integer;
begin
  try
    dIni:=TRegistryIniFile.Create('RtColorPicker');
    with dIni do
      for i:=0 to 15 do
        WriteString('CustomColors',ColorDialog.CustomColors.Names[i],
            ColorDialog.CustomColors.Values[ColorDialog.CustomColors.Names[i]]);
  finally
    dIni.Free;
  end;
end;

procedure TColorPickerForm.SetPickColor(Value:TColor);
var i:integer; ok:boolean;
begin
  FPickColor:=Value;
  ok:=False;
  for i:=1 to NormalColorsCnt+SystemColorsCnt do
  begin
    if BtnColors[i]=FPickColor then
    begin
      ColBtns[i].down:=True;
      Ok:=true;
    end else ColBtns[i].down:=False;
  end;
  if not Ok then
  begin
    OtherColBtn.SqareColor:=FPickColor;
    OtherColBtn.Down:=True;
  end else OtherColBtn.Down:=False;
end;

procedure TColorPickerForm.OtherBtnClick(Sender:TObject);
begin
  FDoingDialog:=True;
  if ColorDialog.Execute then
  begin
    OtherColBtn.SqareColor:=ColorDialog.Color;
    BtnClick(OtherColBtn);
  end;
  FDoingDialog:=False;
end;

procedure TColorPickerForm.BtnClick(Sender:TObject);

begin
  FPickColor:=TColBtn(Sender).SqareColor;
  AssignedColorPickCombo.ActiveColor:=FPickColor;
  Close;
end;

procedure TColorPickerForm.FormDeactivate(Sender: TObject);
begin
  if not FDoingDialog then Self.Close;
end;

procedure TColorPickerForm.FormKeyPress(Sender: TObject; var Key: Char);
begin
  if Key=#27{Esc} then close;
end;

procedure TColorPickCombo.DrawItem(Index:Integer;Rect:TRect;State:TOwnerDrawState);
var B:integer;
begin
  B:=Height div 5;
  with Canvas do
  begin
    Brush.Style:=bsSolid;
    Brush.Color:=Color;
    FillRect(Rect);  // fill the background
    Pen.color:=clgray;
    Brush.Color:=FActiveColor;
    Rectangle(B,B,Height-B,Height-B);
  end;
end;

procedure TColorPickCombo.CreateWnd;
begin
  inherited CreateWnd;
  Items:=FItems;
  ItemIndex:=0;
end;

constructor TColorPickCombo.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  // create form on first call
  if not Assigned(ColorPickerForm) then
  begin
    ColorPickerForm:=TColorPickerForm.Create(Application);
    ColorPickerForm.Close;
  end;
  Height :=22;
  FOtherBtnCaption:='&Other colors...';
  Style:=csOwnerDrawFixed;
  FItems:=TStringList.Create;
  FItems.Add('1');
  OnDropDown:=MyDropDown;
  FDropDnColor:=clBtnFace;
end;

procedure TColorPickCombo.MyDropDown(Sender:TObject);
var P:TPoint;
begin
  P.X:=0;
  P.Y:=Height;
  P:=ClientToScreen(P);
  with ColorPickerForm do
  begin
    if P.X+Width>Screen.Width then P.X:=Screen.Width-Width;
    if P.Y+Height>Screen.Height then P.Y:=P.Y-Height-Self.Height;
    Color:=FDropDnColor;
    Left:=P.X;
    Top:=P.Y;
    AssignedColorPickCombo:=Self;
    PickColor:=ActiveColor;
    OtherBtn.Caption:=FOtherBtnCaption;
    Show;
  end;
end;

procedure TColorPickCombo.AdjustSize (var W: Integer; var H: Integer);
begin
  W:=H+(H div 2)+6;
end;

procedure TColorPickCombo.SetBounds(ALeft, ATop, AWidth, AHeight: Integer);
var
  W, H: Integer;
begin
  W := AWidth;
  H := AHeight;
  AdjustSize (W, H);
 inherited SetBounds (ALeft, ATop, W, H);
end;

procedure TColorPickCombo.SetActiveColor(Value:TColor);
begin
  if Value<>FActiveColor then
  begin
    FActiveColor:=Value;
    if Assigned(FOnChanged) then FOnChanged(Self);
    invalidate;
  end;
end;

end.
